#!/bin/bash
set -e

### Instelbare variabelen #########################################

SERVICE_NAME="api-agent"
SERVICE_USER="root"
SERVICE_GROUP="root"

# Waar de bin en json-configs moeten staan
BIN_DIR="/var/opt/codesys/connector"
BIN_CONFIG_DIR="${BIN_DIR}/config"

# Waar groupdef.xml moet staan
HCS_CONNECTOR_DIR="/var/opt/codesys/HCS/connector"

BINARY_NAME="api-agent.bin"

# Config directory in het installer-mapje (relatief t.o.v. deze script)
INSTALL_CONFIG_DIR="./config"

GROUPDEF_FILE_NAME="groupdef.xml"

# Logbestand – moet overeenkomen met wat je in de bin gebruikt
LOG_FILE="/var/log/user.log"

###################################################################

echo "=== Installer voor ${SERVICE_NAME} (bin-only) ==="

if [[ $EUID -ne 0 ]]; then
    echo "Deze installer moet als root gerund worden (sudo ./install.sh)."
    exit 1
fi

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$SCRIPT_DIR"

# 1) Systeemgebruiker aanmaken (als hij nog niet bestaat)
if id -u "${SERVICE_USER}" >/dev/null 2>&1; then
    echo "Gebruiker ${SERVICE_USER} bestaat al."
else
    echo "Gebruiker ${SERVICE_USER} aanmaken..."
    useradd -r -s /usr/sbin/nologin "${SERVICE_USER}"
fi

# 2) Doelmappen aanmaken
echo "Doelmap voor bin: ${BIN_DIR}..."
mkdir -p "${BIN_DIR}"
chown "${SERVICE_USER}:${SERVICE_GROUP}" "${BIN_DIR}"

echo "Doelmap voor json-configs: ${BIN_CONFIG_DIR}..."
mkdir -p "${BIN_CONFIG_DIR}"
chown "${SERVICE_USER}:${SERVICE_GROUP}" "${BIN_CONFIG_DIR}"

echo "Doelmap voor groupdef.xml: ${HCS_CONNECTOR_DIR}..."
mkdir -p "${HCS_CONNECTOR_DIR}"
chown "${SERVICE_USER}:${SERVICE_GROUP}" "${HCS_CONNECTOR_DIR}"

# 3) Binary kopiëren
if [[ ! -f "./${BINARY_NAME}" ]]; then
    echo "FOUT: Binary ./${BINARY_NAME} niet gevonden in installer-map."
    exit 1
fi

echo "Binary kopiëren naar ${BIN_DIR}/${BINARY_NAME}..."
cp "./${BINARY_NAME}" "${BIN_DIR}/${BINARY_NAME}"
chmod 755 "${BIN_DIR}/${BINARY_NAME}"
chown "${SERVICE_USER}:${SERVICE_GROUP}" "${BIN_DIR}/${BINARY_NAME}"

# 4) Config-bestanden kopiëren
if [[ -d "${INSTALL_CONFIG_DIR}" ]]; then
    echo "JSON-configs en groupdef.xml kopiëren vanuit ${INSTALL_CONFIG_DIR}..."

    # JSON-bestanden naar BIN_CONFIG_DIR
    shopt -s nullglob
    for json in "${INSTALL_CONFIG_DIR}"/*.json; do
        echo " - JSON: $(basename "$json") -> ${BIN_CONFIG_DIR}"
        cp "$json" "${BIN_CONFIG_DIR}/"
        chown "${SERVICE_USER}:${SERVICE_GROUP}" "${BIN_CONFIG_DIR}/$(basename "$json")"
    done
    shopt -u nullglob

    # groupdef.xml naar HCS_CONNECTOR_DIR
    if [[ -f "${INSTALL_CONFIG_DIR}/${GROUPDEF_FILE_NAME}" ]]; then
        echo " - groupdef.xml -> ${HCS_CONNECTOR_DIR}"
        cp "${INSTALL_CONFIG_DIR}/${GROUPDEF_FILE_NAME}" "${HCS_CONNECTOR_DIR}/${GROUPDEF_FILE_NAME}"
        chown "${SERVICE_USER}:${SERVICE_GROUP}" "${HCS_CONNECTOR_DIR}/${GROUPDEF_FILE_NAME}"
    else
        echo "WAARSCHUWING: ${INSTALL_CONFIG_DIR}/${GROUPDEF_FILE_NAME} niet gevonden."
    fi
else
    echo "WAARSCHUWING: config-directory ${INSTALL_CONFIG_DIR} niet gevonden; overslaan."
fi

# 5) Logfile klaarzetten
echo "Logbestand ${LOG_FILE} klaarzetten..."
touch "${LOG_FILE}"
chown "${SERVICE_USER}:${SERVICE_GROUP}" "${LOG_FILE}"
chmod 644 "${LOG_FILE}"

# 6) systemd-servicebestand aanmaken
SERVICE_FILE="/etc/systemd/system/${SERVICE_NAME}.service"

echo "Systemd service in ${SERVICE_FILE} aanmaken..."

cat > "${SERVICE_FILE}" <<EOF
[Unit]
Description=API Agent service
After=network.target

[Service]
Type=simple
User=${SERVICE_USER}
Group=${SERVICE_GROUP}
WorkingDirectory=${BIN_DIR}
ExecStart=${BIN_DIR}/${BINARY_NAME}
Restart=on-failure
RestartSec=5

# Omgevingsvariabelen voor de agent (worden door de bin gelezen)
Environment=GROUPDEF_PATH=${HCS_CONNECTOR_DIR}/${GROUPDEF_FILE_NAME}
# Eventueel later:
# Environment=CONNECTOR_LOG_LEVEL=INFO

[Install]
WantedBy=multi-user.target
EOF

echo "Rechten op servicebestand instellen..."
chmod 644 "${SERVICE_FILE}"

# 7) systemd herladen en service enable/start
echo "systemd daemon herladen..."
systemctl daemon-reload

echo "Service ${SERVICE_NAME} inschakelen bij boot..."
systemctl enable "${SERVICE_NAME}"

echo "Service ${SERVICE_NAME} nu starten..."
systemctl restart "${SERVICE_NAME}"

echo "=== Installatie voltooid ==="
systemctl status "${SERVICE_NAME}" --no-pager || true
